function [xss, yss, sss, Pmatss, H, G, H2, G2, retcode] = ...
    bfor_endogenous_ms_solver_main(f, Pmat, vars, derivs, steady, THETA, nTHETA)
% 
% solves a general endogenous Markov-switching model to the second order
% as in Benigno, Foerster, Otrok, and Rebucci (2020)
% 
% Updated 2020/12
% Benigno, Foerster, Otrok, and Rebucci
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 
% 
% INPUTS
%  f        = (n x 1) vector of symbols with the equilibrium equations
%  Pmat     = (ns x ns) matrix of symbols with the transition matrix
%  vars     = structure with elements that are variables of the system
%  derivs   = structure with elements that are deriatives of f with respect
%               to the variables of the system
%  THETA    = vector of parameters in symbolic form
%  nTHETA   = vector of parameters of numeric parameters to solve the model
% 
% 
% OUTPUTS
%  xss      = (nx x 1) vector of steady state of x, the predetermined 
%               variables
%  yss      = (ny x 1) vector of steady state of y, the nonpredetermined 
%               variables
%  sss      = (ns x 1) vector of erdogic distribution across regimes
%  Pmatss   = (ns x ns) matrix of the transition matrix at steady state
%  H        = (nx x (nx+ne+1) x ns) array for first-order solution for the 
%               predetermined variables, H(i,j,s) is the response of x(i) 
%               to state(j) given regime s, state=[xlag;epsilon;chi]
%  G        = (ny x (nx+ne+1) x ns) array for first-order solution for the 
%               non-predetermined variables, G(i,j,s) is the response of 
%               y(i) to state(j) given regime s, state=[xlag;epsilon;chi]
%  H2       = (nx x (nx+ne+1)^2 x ns) array for second-order solution for 
%               the predetermined variables, H2(i,j,s) is the response of 
%               x(i) to the j-element of kron(state,state) givenregime s
%  G2       = (ny x (nx+ne+1)^2 x ns) array for second-order solution for 
%               the non-predetermined variables, G2(i,j,s) is the response 
%               of y(i) to the j-element of kron(state,state) conditional 
%               on regime s
%  retcode  = indicator taking values
%       0: solution not found, no steady state, or iterative procedure 
%               does not converge
%       1: solution found, iterative procedure converges  
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % 

 
% -- Setup -- %
modelsoln = tic;
ns = length(Pmat);
ny = length(vars.yvars);
nx = length(vars.xvars);
ne = length(vars.eps);
n  = length(f);

% -- Initialization -- %
H   = NaN(nx,nx+ne+1,ns);
G   = NaN(ny,nx+ne+1,ns);
H2  = NaN(nx,(nx+ne+1)^2,ns);
G2  = NaN(ny,(nx+ne+1)^2,ns);


% -- Steady State -- %
[xss, yss, t1ss, that, sss, Pmatss, retcode] = ...
    bfor_endogenous_ms_solver_steadystate(f, Pmat, vars, steady, THETA, nTHETA);
if retcode == 0
    warning('Solver Warning -- Steady State Not Found');
    return; 
end

% -- Numerical Derivatives -- %
numderivs = tic;
[nderivs] = bfor_endogenous_ms_solver_derivatives_numeric(derivs, Pmatss, vars, yss, xss, t1ss, THETA, nTHETA);
disp(['Numerical derivatives time = ' num2str(toc(numderivs))]);

% -- First Order -- %
% Get Matrices
[A,B] = bfor_endogenous_ms_solver_matrices_x(nderivs, ns); 

% Run Iterative Procedure
[gx,hx,retcode] = bfor_endogenous_ms_solver_iterativeprocedure(Pmatss, A, B, ny, nx);
if retcode == 0 
    warning('Solver Warning -- Iterative Procedure Does Not Converge');
    return; 
end
for ii = 1:ns
    H(:,1:nx,ii) = hx{ii};
    G(:,1:nx,ii) = gx{ii};
end

% First order for epsilon
[Ce,De] = bfor_endogenous_ms_solver_matrices_e(nderivs, ns, G);
Cefull = zeros(ns*n,ns*n);
Defull = zeros(ns*n,ne);
for ii = 1:ns
    iiblock = (ii-1)*n+1:ii*n;
    for jj = 1:ns
        Cefull(iiblock,iiblock) = Cefull(iiblock,iiblock) + Pmatss(ii,jj)*Ce{jj,ii};
        Defull(iiblock,1:ne) = Defull(iiblock,1:ne) + Pmatss(ii,jj)*De{jj,ii};
    end
end
esoln = -Cefull\Defull; 
for ii = 1:ns
    H(:,nx+1:nx+ne,ii) = esoln((ii-1)*n+1:(ii-1)*n+nx,1:ne);
    G(:,nx+1:nx+ne,ii) = esoln((ii-1)*n+nx+1:ii*n,1:ne);
end


% First order for Chi
[Cc,Dc] = bfor_endogenous_ms_solver_matrices_c(nderivs, ns, that, G);
Ccfull = zeros(ns*n,ns*n);
Dcfull = zeros(ns*n,1);
for ii = 1:ns
    iiblock = (ii-1)*n+1:ii*n;
    for jj = 1:ns
        jjblock = (jj-1)*n+nx+1:jj*n;
        Ccfull(iiblock,iiblock) = Ccfull(iiblock,iiblock) + Pmatss(ii,jj)*Cc{jj,ii}(1:n,1:n);
        Ccfull(iiblock,jjblock) = Ccfull(iiblock,jjblock) + Pmatss(ii,jj)*Cc{jj,ii}(1:n,n+1:n+ny);
        Dcfull(iiblock,1) = Dcfull(iiblock,1) + Pmatss(ii,jj)*Dc{jj,ii}(1:n,1);
    end
end
csoln = -Ccfull\Dcfull;
for ii = 1:ns
    H(:,nx+ne+1,ii) = csoln((ii-1)*n+1:(ii-1)*n+nx);
    G(:,nx+ne+1,ii) = csoln((ii-1)*n+nx+1:ii*n);
end

% -- Second Order -- %
% Second order for x,x
[Mxx,Nxx] = bfor_endogenous_ms_solver_matrices_xx(nderivs, Pmatss, H, G);
Mxxfull = zeros(ns*(n*nx*nx),ns*(n*nx*nx));
Nxxfull = zeros(ns*(n*nx*nx),1);
for ii = 1:ns
    iiblock = (ii-1)*n*nx*nx+1:ii*n*nx*nx;
    for jj = 1:ns
        jjblock = (jj-1)*n*nx*nx+nx*nx*nx+1:jj*n*nx*nx;
        Mxxfull(iiblock,iiblock) = Mxxfull(iiblock,iiblock) + ...
            Mxx{jj,ii}(:,1:n*nx*nx);
        Mxxfull(iiblock,jjblock) = Mxxfull(iiblock,jjblock) + ...
            Mxx{jj,ii}(:,n*nx*nx+1:(n+ny)*nx*nx);
        Nxxfull(iiblock) = Nxxfull(iiblock) + Nxx{jj,ii};
    end
end
xxsoln = -Mxxfull\Nxxfull;
for ii = 1:ns
    for jj = 1:nx
        H2(:,(jj-1)*(nx+ne+1)+1:(jj-1)*(nx+ne+1)+nx,ii) = reshape(xxsoln((ii-1)*n*nx*nx+(jj-1)*nx^2+1:(ii-1)*n*nx*nx+jj*nx^2),nx,nx);
        G2(:,(jj-1)*(nx+ne+1)+1:(jj-1)*(nx+ne+1)+nx,ii) = reshape(xxsoln((ii-1)*n*nx*nx+nx*nx*nx+(jj-1)*nx*ny+1:(ii-1)*n*nx*nx+nx*nx*nx+jj*nx*ny),ny,nx);
    end
end


% Second order for x,e
[Mxe,Nxe] = bfor_endogenous_ms_solver_matrices_xe(nderivs, Pmatss, H, G, G2);
Mxefull = zeros(ns*n*nx*ne,ns*n*nx*ne);
Nxefull = zeros(ns*n*nx*ne,1);
for ii = 1:ns
	iiblock = (ii-1)*n*nx*ne+1:ii*n*nx*ne;
    for jj = 1:ns
        Mxefull(iiblock,iiblock) = Mxefull(iiblock,iiblock) + Mxe{jj,ii}(:,1:n*nx*ne);
        Nxefull(iiblock) = Nxefull(iiblock) + Nxe{jj,ii};
    end
end
xesoln = -Mxefull\Nxefull;
for ii = 1:ns
    for jj = 1:nx
        H2(:,(jj-1)*(nx+ne+1)+nx+1:(jj-1)*(nx+ne+1)+nx+ne,ii) = reshape(xesoln((ii-1)*n*nx*ne+(jj-1)*nx*ne+1:(ii-1)*n*nx*ne+jj*nx*ne),nx,ne);
        G2(:,(jj-1)*(nx+ne+1)+nx+1:(jj-1)*(nx+ne+1)+nx+ne,ii) = reshape(xesoln((ii-1)*n*nx*ne+nx*nx*ne+(jj-1)*ny*ne+1:(ii-1)*n*nx*ne+nx*nx*ne+jj*ny*ne),ny,ne);
    end
end


% Second order for x,c
[Mxc,Nxc] = bfor_endogenous_ms_solver_matrices_xc(nderivs, that, Pmatss, H, G, G2);
Mxcfull = zeros(ns*n*nx,ns*n*nx);
Nxcfull = zeros(ns*n*nx,1);
for ii = 1:ns
	iiblock = (ii-1)*n*nx+1:ii*n*nx;
	for jj = 1:ns
       jjblock = (jj-1)*n*nx+nx*nx+1:jj*n*nx;
       Mxcfull(iiblock,iiblock) = Mxcfull(iiblock,iiblock) + Mxc{jj,ii}(:,1:n*nx);
       Mxcfull(iiblock,jjblock) = Mxcfull(iiblock,jjblock) + Mxc{jj,ii}(:,n*nx+1:(n+ny)*nx);
       Nxcfull(iiblock) = Nxcfull(iiblock) + Nxc{jj,ii};
	end
end
xcsoln = -Mxcfull\Nxcfull;
for ii = 1:ns
    for jj = 1:nx
        H2(:,(jj-1)*(nx+ne+1)+nx+ne+1,ii) = reshape(xcsoln((ii-1)*n*nx+(jj-1)*nx+1:(ii-1)*n*nx+jj*nx),nx,1);
        G2(:,(jj-1)*(nx+ne+1)+nx+ne+1,ii) = reshape(xcsoln((ii-1)*n*nx+nx*nx+(jj-1)*ny+1:(ii-1)*n*nx+nx*nx+jj*ny),ny,1);
    end
end

% Second order for e,x (use Young's theorem)
for ii = 1:ns
    for jj = 1:ne
        for kk = 1:nx
            H2(:,(nx+jj-1)*(nx+ne+1)+kk,ii) = xesoln((ii-1)*n*nx*ne+(kk-1)*nx*ne+(jj-1)*nx+1:(ii-1)*n*nx*ne+(kk-1)*nx*ne+jj*nx);
            G2(:,(nx+jj-1)*(nx+ne+1)+kk,ii) = xesoln((ii-1)*n*nx*ne+nx*nx*ne+(kk-1)*ny*ne+(jj-1)*ny+1:(ii-1)*n*nx*ne+nx*nx*ne+(kk-1)*ny*ne+jj*ny);
        end
    end
end

% Second order for e,e
[Mee,Nee] = bfor_endogenous_ms_solver_matrices_ee(nderivs, Pmatss, H, G, G2);
Meefull = zeros(ns*n*ne*ne,ns*n*ne*ne);
Neefull = zeros(ns*n*ne*ne,1);
for ii = 1:ns
	iiblock = (ii-1)*n*ne*ne+1:ii*n*ne*ne;
    for jj = 1:ns
        Meefull(iiblock,iiblock) = Meefull(iiblock,iiblock) + Mee{jj,ii}(:,1:n*ne*ne);
        Neefull(iiblock) = Neefull(iiblock) + Nee{jj,ii};    
    end
end
eesoln = -Meefull\Neefull;
for ii = 1:ns
    for jj = 1:ne
        for kk = 1:ne
            H2(:,(nx+jj-1)*(nx+ne+1)+nx+kk,ii) = eesoln((ii-1)*n*ne*ne+(jj-1)*ne*nx+(kk-1)*nx+1:(ii-1)*n*ne*ne+(jj-1)*ne*nx+kk*nx);
            G2(:,(nx+jj-1)*(nx+ne+1)+nx+kk,ii) = eesoln((ii-1)*n*ne*ne+nx*ne*ne+(jj-1)*ne*ny+(kk-1)*ny+1:(ii-1)*n*ne*ne+nx*ne*ne+(jj-1)*ne*ny+kk*ny);
        end
    end
end

% Second order for e,c
[Mec,Nec] = bfor_endogenous_ms_solver_matrices_ec(nderivs, Pmatss, that, H, G, G2);
Mecfull = zeros(ns*n*ne,ns*n*ne);
Necfull = zeros(ns*n*ne,1);
for ii = 1:ns
	iiblock = (ii-1)*n*ne+1:ii*n*ne;
    for jj = 1:ns
        Mecfull(iiblock,iiblock) = Mecfull(iiblock,iiblock) + Mec{jj,ii}(:,1:n*ne);
        Necfull(iiblock) = Necfull(iiblock) + Nec{jj,ii};    
    end
end
ecsoln = -Mecfull\Necfull;
for ii = 1:ns
    for jj = 1:ne
        H2(:,(nx+jj-1)*(nx+ne+1)+nx+ne+1,ii) = ecsoln((ii-1)*n*ne+(jj-1)*nx+1:(ii-1)*n*ne+jj*nx);
        G2(:,(nx+jj-1)*(nx+ne+1)+nx+ne+1,ii) = ecsoln((ii-1)*n*ne+nx*ne+(jj-1)*ny+1:(ii-1)*n*ne+nx*ne+jj*ny);
    end
end


% Second order for c,x (use Young's theorem)
for ii = 1:ns
    for kk = 1:nx
        H2(:,(nx+ne)*(nx+ne+1)+kk,ii) = xcsoln((ii-1)*n*nx+(kk-1)*nx+1:(ii-1)*n*nx+kk*nx);
        G2(:,(nx+ne)*(nx+ne+1)+kk,ii) = xcsoln((ii-1)*n*nx+nx*nx+(kk-1)*ny+1:(ii-1)*n*nx+nx*nx+kk*ny);
    end
end    
    
% Second order for c,e (use Young's theorem)
for ii = 1:ns
    for kk = 1:ne
        H2(:,(nx+ne)*(nx+ne+1)+nx+kk,ii) = ecsoln((ii-1)*n*ne+(kk-1)*nx+1:(ii-1)*n*ne+kk*nx);
        G2(:,(nx+ne)*(nx+ne+1)+nx+kk,ii) = ecsoln((ii-1)*n*ne+ne*nx+(kk-1)*ny+1:(ii-1)*n*ne+ne*nx+kk*ny);
    end
end    

% Second order for cc
[Mcc,Ncc] = bfor_endogenous_ms_solver_matrices_cc(nderivs, that, Pmatss, H , G, G2);
Mccfull = zeros(ns*n,ns*n);
Nccfull = zeros(ns*n,1);
for ii = 1:ns
	iiblock = (ii-1)*n+1:ii*n;
    for jj = 1:ns
        jjblock = (jj-1)*n+nx+1:jj*n;
        Mccfull(iiblock,iiblock) = Mccfull(iiblock,iiblock) + Mcc{jj,ii}(:,1:n);
        Mccfull(iiblock,jjblock) = Mccfull(iiblock,jjblock) + Mcc{jj,ii}(:,n+1:n+ny);
        Nccfull(iiblock) = Nccfull(iiblock) + Ncc{jj,ii};    
    end
end
ccsoln = -Mccfull\Nccfull;
for ii = 1:ns
    H2(:,(nx+ne+1)^2,ii) = ccsoln((ii-1)*n+1:(ii-1)*n+nx);
    G2(:,(nx+ne+1)^2,ii) = ccsoln((ii-1)*n+nx+1:ii*n);
end

% Truncate Imaginary Values
if max(max(imag(H)))  < 1e-10;  H  = real(H); end
if max(max(imag(G)))  < 1e-10;  G  = real(G); end
if max(max(imag(H2))) < 1e-10;  H2 = real(H2); end
if max(max(imag(G2))) < 1e-10;  G2 = real(G2); end


disp(['Model solution time = ' num2str(toc(modelsoln))]);

end